# Copyright (C) 2017 Linaro Limited
#
# Author: Senthil Kumaran S <senthil.kumaran@linaro.org>
#
# SPDX-License-Identifier: GPL-2.0-or-later

import unittest
from unittest.mock import patch

from lava_common.exceptions import JobError
from tests.lava_dispatcher.test_basic import Factory, LavaDispatcherTestCase
from tests.utils import infrastructure_error_multi_paths


class TestDownloadDeploy(LavaDispatcherTestCase):
    def setUp(self):
        super().setUp()
        self.factory = Factory()
        self.job = self.factory.create_job("db410c-01", "sample_jobs/download.yaml")

    def test_deploy_job(self):
        self.assertEqual(self.job.pipeline.job, self.job)
        self.assertIsInstance(self.job.device["device_info"], list)
        for action in self.job.pipeline.actions:
            self.assertEqual(action.job, self.job)

    def test_pipeline(self):
        description_ref = self.pipeline_reference("download.yaml", job=self.job)
        self.assertEqual(description_ref, self.job.pipeline.describe())

    @unittest.skipIf(
        infrastructure_error_multi_paths(["lxc-info", "img2simg", "simg2img"]),
        "lxc or img2simg or simg2img not installed",
    )
    def test_validate(self):
        try:
            self.job.pipeline.validate_actions()
        except JobError as exc:
            self.fail(exc)
        for action in self.job.pipeline.actions:
            self.assertEqual([], action.errors)

    def test_directories(self):
        job = self.factory.create_job("bbb-01", "sample_jobs/download_dir.yaml")
        with self.assertRaises(JobError):
            job.validate()

    def test_download_tar(self):
        job = self.factory.create_job("x86-01", "sample_jobs/up2-tests-from-tar.yaml")
        with patch("lava_dispatcher.actions.deploy.nbd.which") as which_mock:
            job.validate()

        which_mock.assert_any_call("nbd-server")
        which_mock.assert_any_call("in.tftpd")
        self.assertEqual(job.pipeline.errors, [])
        description_ref = self.pipeline_reference("up2-tests-from-tar.yaml", job=job)
        self.assertEqual(description_ref, job.pipeline.describe())
