from __future__ import print_function
from __future__ import unicode_literals

import datetime
import json
from builtins import str
from collections import OrderedDict
from re import match
from typing import Optional, Dict, Any
from urllib.parse import urlparse, parse_qs

import requests
from future import standard_library
from isoweek import Week
from past.builtins import basestring

import sugarcrm.sugarcrm as sugarcrm
from pcs_proxy.common import settings, debug_log
from pcs_proxy.server_common import CollaboraServiceProxyLogic, AuthenticationType

standard_library.install_aliases()


class Forecast(sugarcrm.SugarObject):
    module = "ColFo_CollaboraForecast"


sugarcrm.Forecast = Forecast


class Invoice(sugarcrm.SugarObject):
    module = "ColIn_CollaboraInvoices"


sugarcrm.Invoice = Invoice


class Project(sugarcrm.SugarObject):
    module = "Project"


sugarcrm.Project = Project


class Opportunity(sugarcrm.SugarObject):
    module = "Opportunity"


sugarcrm.Opportunity = Opportunity


class CollaboraServiceProxyLogicCRM(CollaboraServiceProxyLogic):
    """
    PCS-to-CRM handler
    """
    service_name = "CRM"
    service_hostname = "crm.collabora.com"
    service_api = "wire.collabora.com"

    # Update fields in CRM for a given Project module entry
    #
    # NOTE: this is basically a "patch" update, so if you wish to clear a given
    # field, it should be passed in for update as a blank string
    #
    # project_code  str()   the project code to apply the updates to
    # records       dict()  a [key]=>[value] dict of new field values to apply
    def update_project_record(self, project_id, records):
        debug_log(" *** Calling update_project_record ***")
        #FIXME: nasty hack...
        if not records:
            debug_log('No records, early out')
            return
        debug_log("Going ahead with update_module_record with records: " + str(records))
        project = self.sugar.get_entry('Project', project_id)
        debug_log('Found project: ' + str(vars(project)))
        vars(project).update(records)
        self.sugar.set_entry(project)
        debug_log('Set project: ' + str(vars(project)))

    # Update fields Opportunity in CRM for a given project_id module entry
    #
    # NOTE: this is basically a "patch" update, so if you wish to clear a given
    # field, it should be passed in for update as a blank string
    #
    # project_id  str()   the project id to apply the updates to
    # records     dict()  a [key]=>[value] dict of new field values to apply
    def update_opportunity_record(self, project_id, records):
        #FIXME: nasty hack...
        if not records:
            debug_log('No records, early out')
            return
        project_opportunities = self.sugar.get_entry(
            'Project',
            project_id,
            dict(opportunities=['id'])
        )
        if hasattr(project_opportunities, 'opportunities'):
            # Check there is a single opportunity assigned to the project, as expected.
            assert len(project_opportunities.opportunities) == 1

            opportunity = self.sugar.get_entry('Opportunities', str(project_opportunities.opportunities[0].id))
            debug_log('Found opportunity: ' + str(vars(opportunity)))
            vars(opportunity).update(records)
            self.sugar.set_entry(opportunity)
            debug_log('Set opportunity: ' + str(vars(opportunity)))
        else:
            debug_log('No opportunity found for this project.')

    def validate_project_id(self, project_id, project_code):
        project = self.sugar.get_entry('Project', project_id)
        if project is None:
            return False, 'Project with ID {} was not found.'.format(project_id)
        elif project.project_code_c != project_code:
            return False, 'Project with ID {} has code {}, not {}'.format(
                project_id,
                project.project_code_c,
                project_code
            )
        return True, project

    # Check the request URI received by the proxy to see if it's validly formed
    # and meets various requirements for our PCS => CRM requests
    #
    # Returns a human-readable string of "OK" if the request validates
    # Returns a human-readable error string otherwise
    #
    # validate_URI  str()  the request URI to be checked for validity
    def prevalidate_request(self, validate_URI):
        debug_log("Prevalidating request for URI:\n  "+validate_URI)

        query = parse_qs(urlparse(validate_URI).query)

        if not 'record_id' in query:
            return "Project ID is empty or missing from the request. Check 'Health' tab."

        project_id = query['record_id'][0]
        project_code = query['proj_code'][0]

        debug_log("PCS provides project code:\n"+project_code)

        # new 3 step validation from nmilner
        if project_id:
            tested_OK, _ = self.validate_project_id(project_id, project_code)

            if not tested_OK:
                return "The project ID you have configured does not match the project code provided!"
        else:
            return "You have not configured a project ID!"

        return "OK"

    # A "blind fire" function that sees if the stored CRM API login token is not
    # expired, and if it is expired, renews it
    def session_ensure(self):
        self.sugar = sugarcrm.CollaboraSession(
            settings['ProxyLogicProvider'].request_URL(),
            settings['credo_u'],
            settings['credo_p']
        )

    # Turn the results into a dictionary for easier querying.
    def _dictify_results(self, results, key_field='name'):
        results_dict = {}
        for x in results:
            results_dict[getattr(x, key_field)] = x
        return results_dict

    # Set the relationships between two sets of CRM records
    #
    # Returns the result of the API call,
    #
    # entry_ids    list()           the entity ids that are being related to
    # rel_idx      list() or str()  the entity id (str) or ids(list) that are the
    #                               "basis" of the created relationship
    # relating_to  str()            the non-Project module name these records are
    #                               associated with
    def set_relationships(self, entry_ids, rel_idx, relating_to, module_name='Project'):
        rest_data_dict = OrderedDict()
        rest_data_dict['session'] = self.sugar.session_id
        rest_data_dict['module_names'] = [module_name] * len(entry_ids)
        if not isinstance(rel_idx, basestring):
            rest_data_dict['module_ids'] = rel_idx
        else:
            rest_data_dict['module_ids'] = [rel_idx] * len(entry_ids)
        rest_data_dict['link_field_names'] = [relating_to] * len(entry_ids)
        rest_data_dict['related_ids'] = [entry_ids]
        rest_data_dict['name_value_lists'] = [OrderedDict()] * len(entry_ids)
        rest_data_dict['delete'] = 0
        return self.sugar._request('set_relationships', rest_data_dict)

    def handle_request(self, request_str):
        query = parse_qs(urlparse(request_str).query)

        module = query['the_module_to_update'][0]
        project_id = None
        project_code = None

        if 'record_id' in query:
            project_id = query['record_id'][0]
            project_code = query['proj_code'][0]

        data_file = query['data_file'][0]
        data = json.load(open(data_file))

        if not project_id:
            project_id = data['crm_project_id']
            project_code = data['project_code']

        tested_OK, project = self.validate_project_id(project_id, project_code)
        if not tested_OK:
            return "The project ID you have configured does not match the project code provided!"

        if module == 'Project':
            return self.handle_project_request(query, data, project)

        if module == 'Forecast':
            return self.handle_forecast_request(query, data, project)

        if module == 'Invoicing':
            return self.handle_invoice_request(query, data, project)

        if module == 'Forecast_Actuals':
            return self.handle_forecast_actuals_request(query, data, project)

        return 'Unknown module!'

    def handle_forecast_actuals_request(self, query, data, project):
        assert query['the_module_to_update'][0] == 'Forecast_Actuals'
        auth = (settings['credo_u'], settings['credo_p'])
        response = requests.post("https://%s/pcs_hours" % settings['ProxyLogicProvider'].service_api, json=data, auth=auth)
        return response._content

    def handle_forecast_request(self, query, data, project):
        assert query['the_module_to_update'][0] == 'Forecast'

        today = datetime.date.today()
        week = Week.withdate(today)
        this_monday = week.monday().strftime('%Y-%m-%d')

        # Obtain all existing forecasts
        project_forecasts = self.sugar.get_entry(
            'Project',
            project.id,
            dict(colfo_collaboraforecast_project=['id', 'from_date', 'forecast_type_c'])
        )
        if hasattr(project_forecasts, 'colfo_collaboraforecast_project'):
            # We don't want to delete past forecasts
            forecasts_ids = [
                x.id
                for x in project_forecasts.colfo_collaboraforecast_project
                if (x.from_date > this_monday and x.forecast_type_c == 'Project')
                or x.forecast_type_c == 'Project-Monthly-Revenue-FTE']
        else:
            forecasts_ids = []
        forecasts = self.sugar.get_entries('ColFo_CollaboraForecast', forecasts_ids)

        # Check if there is not any forecast missing
        assert len(forecasts_ids) == len(forecasts)

        existing_forecast_names = [ f.name for f in forecasts ]

        # Set them all to deleted at first, so that any that are gone from the
        # spreadsheet get deleted. Those we find we will reinstate.
        for f in forecasts:
            f.deleted = True

        users = self.get_all_crm_users()
        for item in data:
            assert item['the_module_to_update'] == 'Forecast'
            item = item['data']
            # We don't want to push past forecasts
            if item['from_date'] < this_monday and item['forecast_type_c'] == 'Project':
                continue

            monthly_forecast = False
            try:
                if item['forecast_type_c'] == 'Project-Monthly-Revenue-FTE':
                    forecast_data = item['summary'][0]['monthly-revenue-fte']
                    monthly_forecast = True
                else:
                    forecast_data = item['summary'][0]['forecast']
            except TypeError:
                return 'Malformed summary for {}'.format(item['name'])

            del item['summary']

            if not monthly_forecast:
                name = item['name']
                for entry in forecast_data:
                    user = entry['username']
                    del entry['username']
                    item['name'] = name + '-' + user
                    if 'revenue' in entry:
                        item['amount'] = entry['revenue']
                        del entry['revenue']
                    try:
                        item['assigned_user_id'] = users[user].id
                    except KeyError:
                        return 'User {} does not exist'.format(user)
                    item.update(entry)

                    if not item['name'] in existing_forecast_names:
                        forecast = sugarcrm.Forecast(**item)
                        forecasts.append(forecast)
                    else:
                        forecast = [ f for f in forecasts if f.name == item['name'] ][0]
                        vars(forecast).update(item)
                    forecast.currency_id = project.currency_id
                    forecast.deleted = False
            else:
                for k, v in list(forecast_data[0].items()):
                    # CRM fields end up with "_c" and use "_" instead of "-"
                    if k.startswith('fte-'):
                        item[k.replace('-','_') + '_c'] = float(v)
                    elif k == 'revenue':
                        item['amount'] = float(v)
                if not item['name'] in existing_forecast_names:
                    forecast = sugarcrm.Forecast(**item)
                    forecasts.append(forecast)
                else:
                    forecast = [ f for f in forecasts if f.name == item['name'] ][0]
                    vars(forecast).update(item)
                forecast.currency_id = project.currency_id
                forecast.deleted = False

        if forecasts:
            forecasts = self.sugar.set_entries(forecasts)
            associate_forecasts = [f.id for f in forecasts if not f.deleted]

            # Associate the new forecasts to the project
            if associate_forecasts:
                results_dict = settings['ProxyLogicProvider'].set_relationships(associate_forecasts, project.id, "colfo_collaboraforecast_project")
                fail_txt = "\n" + "One or more forecasts failed to create/update, please review any messages above for further details!"

                if 'failed' in results_dict:
                    if int(results_dict['failed']) > 0:
                        return fail_txt

        try:
            data = {
                "forecast_last_updated_c": datetime.datetime.now().strftime('%Y-%m-%d %I:%M:%S')
            }
            self.update_project_record(project.id, data)
        except RuntimeError:
            return "WARNING: Forecasts updated but setting project last_updated value failed"

        return "The forecasts were created/updated successfully!"

    def handle_invoice_request(self, query, data, project):
        assert query['the_module_to_update'][0] == 'Invoicing'

        # Obtain all existing invoices
        invoices = sugarcrm.Invoice(name="{}-%".format(project.project_code_c))
        invoices = self.sugar.get_full_entry_list(invoices)
        existing_invoice_names = [ f.name for f in invoices ]

        fail_log = ''
        for item in data:
            assert item['the_module_to_update'] == 'Invoicing'
            item = item['data']
            if hasattr(project, 'assigned_user_id'):
                item['assigned_user_id'] = project.assigned_user_id
            if item['name'] in existing_invoice_names:
                invoice = [ f for f in invoices if f.name == item['name'] ][0]

                if invoice.invoice_status != "1_not_ready":
                    fail_log = fail_log + 'Can\'t update ' + item['name'] + ', not in "not ready" state!' + "\n"
                    continue
                vars(invoice).update(item)
                settings['lastInfo'] = "Updated \"" + item['name'] + "\""
            else:
                invoice = sugarcrm.Invoice(**item)
                invoices.append(invoice)
                settings['lastInfo'] = "Created \"" + item['name'] + "\""

            invoice.currency_id = project.currency_id

        invoices = self.sugar.set_entries(invoices)
        associate_invoices = [f.id for f in invoices]

        results = settings['ProxyLogicProvider'].set_relationships(associate_invoices, project.id, "colin_collaborainvoices_project")
        if results['failed'] or fail_log:
            return fail_log

        return "The invoices were created/updated successfully!"

    def handle_project_request(self, query, data, project):
        assert query['the_module_to_update'][0] == 'Project'
        project_data = {}
        opportunity_data = {}
        current_module = ''
        for dataset in data:
            # Remove quotes from numbers
            number_pattern = '^(\-)?\d+(\.\d+)?(E\-\d+)?$'
            for key in list(dataset['data'].keys()):
                try:
                    if match(number_pattern, dataset['data'][key]):
                        dataset['data'][key] = float(dataset['data'][key])
                except Exception as error:
                    print(error)

            if dataset['the_module_to_update'] == 'Project':
                project_data.update(dataset['data'])
            elif dataset['the_module_to_update'] == 'Opportunities':
                opportunity_data.update(dataset['data'])
            else:
                print('File {} is malformed!'.format('data_file'))
                assert False

        self.update_project_record(project.id, project_data)
        self.update_opportunity_record(project.id, opportunity_data)

        return 'Request OK'

    def parse_request_str(self, request_str: str, final_pass: bool = False) -> Optional[Dict[str, Any]]:
        assert False

    # Returns a dict containing CRM users
    #
    # The returned dict has dict items of the following format:
    #   [CRM_username] => sugarcrm.User
    def get_all_crm_users(self):
        users = sugarcrm.User()
        results = self.sugar.get_full_entry_list(users)
        return self._dictify_results(results, key_field='user_name')

    @staticmethod
    def build_outgoing_requests(handler):
        url = urlparse(handler.path)
        query = parse_qs(url.query)
        try:
            data = json.load(open(query['data_file'][0]))
        except ValueError:
            return None

        modules = set()
        try:
            mode = data.get('mode')
        except:
            mode = None

        if mode:
            mode = data['mode']
            modules.add('Forecast_Actuals')
        else:
            for record in data:
                for key, val in list(record.items()):
                    if key == "the_module_to_update":
                        modules.add(val)

        returns = []
        for module in modules:
            if module in ('Project', 'Forecast', 'Invoicing', 'Forecast_Actuals'):
                returns.append("/?the_module_to_update={}".format(module))
            elif module == 'Opportunities':
                # Opportunities is handled along with Project, so we do not make
                # a request specifically for it.
                pass
            else:
                print('Unknown module: {}!'.format(module))

        return returns

    # Return the URL base for API calls
    @staticmethod
    def request_URL():
        url = settings['ProxyLogicProvider'].service_hostname+"/service/v4_1/rest.php"
        if not url.startswith('http://') and not url.startswith('https://'):
            url = "https://" + url
        return url

    # CRM proxy runs on TCP/8001
    @staticmethod
    def port_TCP():
        return 8001

    # All CRM API requests use POST
    @staticmethod
    def request_method():
        return "POST"

    # Act on the result received from processing a CRM API request or a group of
    # same.  The result is then passed back down through the proxy and then back
    # down to the PCS macro
    #
    # Returns bytes() with the current chunk of the macro-compatible response
    #
    # msg  str()  the result received from our processing of the macro's request
    # rs   str()  the query string originally sent from the macro to the proxy
    # bh   bool   If True, prepend return with the start of docbook boilerplate
    # eh   bool   If True, append return with the end of docbook boilerplate
    @staticmethod
    def result_filter(msg, rs, bh, eh):
        debug_log(">>> RESULT FILTERING >>>")

        #FIXME: this chunk of code is currently wired just to handle T3958
        # ^ ^ ^ it will need to be more flexible in the future

        query = parse_qs(urlparse(rs).query)
        rec_id = query.get('record_id', ['0'])[0]

        debug_log("Got record ID: " + rec_id)
        debug_log("Got rs: " +str(rs))

        try:
            jsondict = json.loads(msg)

            debug_log("Result from URL " + rs)

            msg_work = ""

            is_project = "the_module_to_update=Project" in rs
            if is_project:
                msg_work = "ERROR: project ID doesn't match code, please check entries on the PCS sheet and try again!"

            try:
                debug_log('Looking for ' + str(rec_id) + " in: " + str(jsondict))
                if is_project:
                    for item in jsondict['entry_list']:
                        for key, val in list(item.items()):
                            if key == "id":
                                debug_log("System asserts ID: " + str(val))

                                if val == rec_id:
                                    msg_work = "The project's record was successfully updated!"
                else:
                    if query['the_module_to_update'][0] != "Opportunity":
                        msg_work = "Request OK"
            except KeyError:
                pass
        except Exception:
            msg_work = msg

            if settings['lastFail'] != "":
                msg_work = settings['lastFail']

        returns = ""

        if bh:
            returns = returns + '<article lang="">' + "\n<para>"

        if msg_work == "Request OK" and not settings['lastInfo'] == "":
            msg_work = msg_work + ": " + settings['lastInfo']

        # msg_work can be received as bytes so convert to str.
        if isinstance(msg_work, bytes):
            msg_work = msg_work.decode('utf-8')
        returns = returns + msg_work

        if eh:
            returns = returns + "</para>\n" + '</article>'

        debug_log(">>> FILTERED RESULT >>>")
        result = returns.replace('&lt;', '<').replace('&gt;', '>')
        debug_log(result)

        # Proxy expects bytes string.
        return result.encode('utf-8')

    def should_retry_request_after_reauthentication(self, response_data) -> bool:
        """
         Check the results of a CRM API request and determine if the response was
        indicating an API token expiry

        Returns True if the API token had expired, this indicates that this
        function refreshed the API token and that the original request should be
        repeated.

        Returns False if the API token appeared to be OK, it should therefore be
        safe to process the response_data as if it was successfully retrieved

        :param response_data: dict()  a JSON-decrypted response from the CRM API
        """
        try:
            parsed = json.loads(response_data)
            s = str(parsed['number'])
            if s == '10' or s == '11':
                debug_log("Invalid session ID, reauth, try again . . .")
                settings['ProxyLogicProvider'].session_ensure()

                # Invalid session ID (expired?)
                return True
        except KeyError:
            pass

        return False

    def get_authentication_type(self) -> AuthenticationType:
        return AuthenticationType.BASIC
