from __future__ import print_function
from __future__ import unicode_literals

import time
from abc import abstractmethod
from builtins import object
from typing import Optional, Dict, Any

from pcs_proxy import AuthenticationType
from pcs_proxy.common import settings

"""
Common code shared by Collabora PCS-to-* services
(other classes derive from this, it isn't used directly)
"""


class CollaboraServiceProxyLogic(object):

    user_agent_timing = 0
    service_name = "Base Service (should never see this)"
    login_session = ""
    service_hostname = ""

    def __init__(self):
        """
        In this case, all we do is mark when the user should be forced to re-enter
        their credentials due to inactivity
        """
        self.user_agent_timing = int(time.time()) + 3600

    def prevalidate_request(self, validate_URI) -> str:
        """
        This is a function that subclassing classes can leverage to validate or
        # reject the validity of a requested URL
        :param validate_URI:
        :return: a status as a string
        """
        return "OK"

    def session_ensure(self) -> None:
        """
        This is a function that subclassing classes can leverage to refresh API
        tokens if they are potentially expired
        """
        pass

    @abstractmethod
    def get_authentication_type(self) -> Optional[AuthenticationType]:
        """
        :return: the authentication type required by the service. None should be used when it is not required.
        """
        pass

    @abstractmethod
    def should_retry_request_after_reauthentication(self, response_data) -> bool:
        pass

    def reconfirm_basic_credentials(self) -> None:
        """
        Checks if the user has left the PCS proxy idle for a while, if they have we
        want to verify that they really want to start working with the proxy again
        and if they do we also ask them to revalidate their credentials
        """

        # it should only exit once we have valid-looking credentials
        if (int(time.time())) > self.user_agent_timing:
            confirms = "n"
            while not confirms.lower().strip() == "y":
                print("You haven't sent data to " + self.service_name + " in a while")
                if settings['Authenticator'].keyring_data:
                    confirms = settings['Authenticator'].get_nonblank_user_input("Are you sure you want to continue (y/n)? ")
                    if not confirms.lower().strip() == "y":
                        print("You must confirm this to continue!")
                else:
                    failure = False
                    cred_ok = False
                    while not cred_ok:
                        if failure:
                            print("Sorry, try again.\n")
                        print("Please confirm your credentials:")
                        failure = (settings['Authenticator'].reconfirm_credentials() is not True)
                        cred_ok = not failure
                    break
        else:
            pass
        self.user_agent_timing = int(time.time()) + 3600

    @abstractmethod
    def parse_request_str(self, request_str: str) -> Optional[Dict[str, Any]]:
        """
        Transform the URL string request_str into a dict with the keys=>values
        """
        pass

    def get_request_headers(self) -> Optional[Dict[str, Any]]:
        """
        :return: optionally returns headers for the requests
        """
        pass
