from __future__ import unicode_literals

import urllib.request
from typing import Any, Dict, Optional

from pcs_proxy.common import settings, URL_PATTERN
from pcs_proxy.server_common import CollaboraServiceProxyLogic, AuthenticationType


class CollaboraServiceProxyLogicChronophage(CollaboraServiceProxyLogic):
    """
    PCS-to-Chronophage handler
    """

    service_name = "Chronophage"
    service_hostname = "chronophage.collabora.com"

    # clean_request_str in this case needn't do anything beyond return the input
    @staticmethod
    def clean_request_str(request_str):
        return request_str

    def handle_request(self, request_str):
        return None

    def parse_request_str(self, request_str: str) -> Optional[Dict[str, Any]]:
        try:
            raw_str = request_str.split("?", 1)[1]
        except IndexError:
            return {}

        fields = raw_str.split("&")
        returns = {}
        for field in fields:
            key = field.split('=', 1)[0]
            val = field.split('=', 1)[1]
            returns[key] = val

        return returns

    # For Chronophage, we need only return the path portion of the raw_str
    @staticmethod
    def build_outgoing_requests(raw_str):
        return [raw_str.path]

    # Return the URL base for API calls
    @staticmethod
    def request_URL():
        base_url = f"{settings['ProxyLogicProvider'].service_hostname}/rest-api"
        return base_url if URL_PATTERN.match(base_url) else f'https://{base_url}'

    # Return the port number the proxy runs on
    @staticmethod
    def port_TCP():
        return 8000

    # All of our comms with Chronophage are done over HTTP GET
    @staticmethod
    def request_method():
        return "GET"

    # We don't need to transform results received from Chronophage, return as-is
    @staticmethod
    def result_filter(msg, rs, bh, eh):
        return msg

    def should_retry_request_after_reauthentication(self, response_data) -> bool:
        """
        We can always count on Chronophage to react nicely to auth-related queries
        """
        return False

    def get_authentication_type(self) -> AuthenticationType:
        return AuthenticationType.TOKEN

    def get_request_headers(self) -> Optional[Dict[str, Any]]:
        return {"Authorization": f"Token {settings.get('apikey', '').strip()}"}

    def session_ensure(self) -> None:
        """
        Tries to reach the Chronophage's health checking endpoint to ensure the API Key is valid
        :return:
        """
        health_check_url = f'{self.request_URL()}/health'
        request = urllib.request.Request(health_check_url, headers=self.get_request_headers())
        with urllib.request.urlopen(request) as response:
            if response.getcode() == 403:
                raise Exception("Forbidden access (HTTP 403)")
