import getpass
from enum import Enum

import gi
from gi.repository.GLib import GError


class AuthenticationType(Enum):
    BASIC = 0
    TOKEN = 1


class Authenticator(object):
    """
    Authentication toolkit below

    This class handles authentication and stores credentials either in the keyring
    if it is available, or in memory otherwise.
    """
    # Try to use LibSecret if available
    try:
        gi.require_version('Secret', '1')
        from gi.repository import Secret
        # This is defined by libsecret for migration from gnome-keyring, but it
        # isn't exported by gobject-introspection, so we redefine it here.
        SECRET_SCHEMA_COMPAT_NETWORK = Secret.Schema.new(
            "org.gnome.keyring.NetworkPassword",
            Secret.SchemaFlags.NONE,
            {
                "user": Secret.SchemaAttributeType.STRING,
                "domain": Secret.SchemaAttributeType.STRING,
                "object": Secret.SchemaAttributeType.STRING,
                "protocol": Secret.SchemaAttributeType.STRING,
                "port": Secret.SchemaAttributeType.INTEGER,
                "server": Secret.SchemaAttributeType.STRING,
                "authtype": Secret.SchemaAttributeType.STRING,
            }
        )
    except ImportError:
        print("LibSecret not found. You will not be able to use the password keyring.")
        Secret = None

    def __init__(self):
        self.keyring_ok = False
        self.keyring_data = None
        # in-RAM stored username
        self.stored_u = ""
        # in-RAM stored password
        self.stored_p = ""

    def find_in_keyring(self, uri):
        """Attempts to load a username and password from the keyring, if the
        keyring is available"""
        if self.Secret is None:
            return

        username = None
        password = None

        try:
            attrs = {
                "domain": uri,
                "server": uri,
                "protocol": "https",
            }
            service = self.Secret.Service.get_sync(0, None)
            # This doesn't give us the password; only details from the schema
            results = service.search_sync(self.SECRET_SCHEMA_COMPAT_NETWORK,
                                          attrs, 0, None)
            if results:
                username = results[0].get_attributes()['user']
                # This gives us only the password
                password = self.Secret.password_lookup_sync(
                    self.SECRET_SCHEMA_COMPAT_NETWORK, attrs, None)
                # Data is already saved in keyring
                self.keyring_data = (username, password)

            # Keyring can be used
            self.keyring_ok = True
        except GError as e:
            # Couldn't contact daemon, or other errors
            print(str(e))
            self.keyring_ok = False

    def save_to_keyring(self, uri, username, password):
        try:
            attrs = {
                "user": username,
                "domain": uri,
                "server": uri,
                # BUG: Passing 'None' for a string causes a segfault
                # https://bugzilla.gnome.org/show_bug.cgi?id=685394
                "object": "",
                "protocol": "https",
            }
            self.Secret.password_store_sync(
                self.SECRET_SCHEMA_COMPAT_NETWORK, attrs,
                self.Secret.COLLECTION_DEFAULT, "PCS-Proxy password",
                password, None)
        except GError as e:
            # Couldn't contact daemon, or other errors
            print(str(e))

    # Store the baseline credentials for accessing downstream resources.
    #
    # u_app  str()  the username to use for downstream access
    # p_app  str()  the password to use for downstream access
    def set_baseline_credentials(self, u_app, p_app):
        self.stored_u = u_app
        self.stored_p = p_app

    # Prompt a user for input, and keep prompting until they provide some
    #
    # Returns the string value of their input (not stripped)
    #
    # prompt    str()    what to prompt the user with
    # password  boolean  If True, disable TTY echoing of user's typed characters
    def get_nonblank_user_input(self, prompt, password=False):
        rv_v = ""
        first = True

        while not rv_v.strip():

            if not first:
                print("Invalid, please try again!")
            else:
                first = False

            if not password:
                rv_v = input(prompt)
            else:
                rv_v = getpass.getpass(prompt)

        return rv_v

    # Get typed username and password from the user
    #
    # Returns a two-item list() with the following format:
    # [str(username), str(password)]
    def gather_username_and_password(self):
        l_u = self.get_nonblank_user_input('Username: ')
        l_p = getpass.getpass("Password for %s: " % (l_u))

        return [l_u, l_p]

    def gather_apikey(self):
        return self.get_nonblank_user_input('API Key: ')

    # Ask the user to re-confirm their entered credentials after a period of
    # inactivity
    #
    # Returns True if the entered credentials match those stored previously in
    #   RAM
    # Returns False otherwise
    def reconfirm_credentials(self):
        (l_u, l_p) = self.gather_username_and_password()

        return l_u == self.stored_u and l_p == self.stored_p
