#!/usr/bin/env python3
from __future__ import print_function
from __future__ import unicode_literals

import http.server
import os
import sys
import urllib.error
import urllib.error
import urllib.parse
import urllib.parse
import urllib.request
import urllib.request
from argparse import ArgumentParser
from builtins import str
from functools import partial
from typing import Optional

from future import standard_library

from pcs_proxy.auth import AuthenticationType, Authenticator
from pcs_proxy.common import settings, debug_log
from pcs_proxy.server_common import CollaboraServiceProxyLogic

standard_library.install_aliases()


class CollaboraServiceProxyHandler(http.server.BaseHTTPRequestHandler):

    def __init__(self, proxy_logic, settings, *args, **kwargs):
        self.settings = settings
        self.proxy_logic = proxy_logic

        # Setup for handling outbound connections
        self.password_manager = urllib.request.HTTPPasswordMgrWithDefaultRealm()

        http_handlers = []
        if self.settings['ProxyLogicProvider'].get_authentication_type() == AuthenticationType.BASIC:
            http_handlers.append(urllib.request.HTTPBasicAuthHandler(self.password_manager))

        self.http_opener = urllib.request.build_opener(*http_handlers)
        super().__init__(*args, **kwargs)

    def handle_request(self, requests, response):
        on_index = 0
        headers = self.settings['ProxyLogicProvider'].get_request_headers()

        for req_item in requests:
            on_index = on_index + 1
            requesting = True
            self.settings['lastFail'] = ""
            self.settings['lastInfo'] = ""

            while requesting:
                rs = (self.proxy_logic.request_URL() + self.path)

                if not rs.endswith(req_item.lstrip("/?")):
                    rs = rs + "&" + req_item.lstrip("/?")

                response = self.settings['ProxyLogicProvider'].handle_request(rs)
                if response is not None:
                    requesting = False
                    result = response
                    continue

                method = self.settings['ProxyLogicProvider'].request_method()

                if method == "POST":
                    debug_log("\nRequest of POST type going out to:\n  " + rs + "\n\n")

                    try:
                        data = urllib.parse.urlencode(self.settings['ProxyLogicProvider'].parse_request_str(rs, True))
                    except TypeError:
                        # This allows us to deal with parse_request_str failures below...
                        data = None

                    real_req = urllib.request.Request(rs, headers=headers, data=data)
                elif method == "GET":
                    debug_log("\nRequest of GET type going out to:\n  " + rs + "\n\n")
                    real_req = urllib.request.Request(rs, headers=headers)
                else:
                    print("\nRequest type " + str(method) + " is not supported\n\n")
                    return

                requesting = False

                if self.settings['lastFail'] == "":
                    try:
                        request = self.http_opener.open(real_req)
                    except Exception as e:
                        request = e

                    result = request.read()
                    requesting = self.settings['ProxyLogicProvider'].should_retry_request_after_reauthentication(result)

                    if self.settings['do_record'] and not requesting:
                        with open('transfer-log', 'a') as f:
                            output = ['>>>>']
                            output.append(real_req.get_full_url())
                            output.append(real_req.get_data())
                            output.append('<<<<')
                            output.append(u'{}\n'.format(str(result)))
                            f.write(u'\n'.join(output).encode('utf-8'))
                else:
                    result = self.settings['lastFail']

            if on_index == 1:
                response = 200

                try:
                    response = request.getcode()
                except UnboundLocalError:
                    # Handle special error cases from parse_request_str
                    pass

                self.send_response(response)
                self.end_headers()

            if on_index == 1:
                if on_index != len(requests):
                    self.wfile.write(self.settings['ProxyLogicProvider'].result_filter(result, rs, True, False) + b"\n")
                else:
                    self.wfile.write(self.settings['ProxyLogicProvider'].result_filter(result, rs, True, True) + b"\n")
                debug_log("\nRESULT PART (added to parsed message):\n")
                debug_log(result)
            else:
                if on_index == len(requests):
                    self.wfile.write(self.settings['ProxyLogicProvider'].result_filter(result, rs, False, True) + b"\n")
                    debug_log("\nRESULT PART (added to parsed message):\n")
                    debug_log(result)
                else:
                    self.wfile.write(self.settings['ProxyLogicProvider'].result_filter(result, rs, False, False) + b"\n")
                    debug_log("\nRESULT (the parsed message):\n")
                    debug_log(result)

    # Handle HTTP server GET requests
    def do_GET(self):
        if settings['ProxyLogicProvider'].get_authentication_type() == AuthenticationType.BASIC:
            self.password_manager.add_password(None, self.proxy_logic.request_URL(), self.settings['credo_u'],
                                               self.settings['credo_p'])

        self.settings['ProxyLogicProvider'].session_ensure()

        self.protocol_version = 'HTTP/1.1'
        requests = self.settings['ProxyLogicProvider'].build_outgoing_requests(self)

        if requests is None:
            prevalidation_result = 'Incompatible PCS: you may need to updated the PCS macros'
        else:
            prevalidation_result = 'OK'

        response = 200

        if prevalidation_result != "OK":
            self.send_response(response)
            self.end_headers()
            self.wfile.write(self.settings['ProxyLogicProvider'].result_filter(prevalidation_result, self.proxy_logic.request_URL(), True, True))
        else:
            self.handle_request(requests, response)


def main(return_server: bool = False, module: Optional[str] = None) -> Optional[http.server.HTTPServer]:
    # Command line argument interface
    help_msg = sys.argv[0] + ": smart proxy server for Collabora internal tools. "

    epilog = """
    Each module has a built-in default server address to use, but you may
    pass a different SERVER (hostname) to use if you are testing changes.
    """

    cli_parser = ArgumentParser(description=help_msg, epilog=epilog)
    cli_parser.add_argument('-d', '--debug', action='store_true',
                            help="Enable output of lots of useful debugging data")
    cli_parser.add_argument('-r', '--record', action='store_true',
                            help="Record requests and responses in a 'transfer-log' file")
    cli_parser.add_argument('-u', '--username', type=str,
                            help="Username (useful for testing)")
    cli_parser.add_argument('-p', '--password', type=str,
                            help="Password (useful for testing)")
    cli_parser.add_argument('-a', '--apikey', type=str,
                            help="API Key (useful for testing)")
    cli_parser.add_argument('--port', type=int, help="Alternate TCP port for the proxy")

    if not module:
        cli_parser.add_argument('module', choices=['crm', 'chronophage'])

    cli_parser.add_argument('server', type=str, nargs='?', help="Alternate server")

    cli_parser.add_argument('api_server', type=str, nargs='?', help="Alternate API server")
    args = cli_parser.parse_args()

    if module:
        args.module = module

    CollaboraServiceProxyLogic = None
    # bring in the appropriate proxying module ("PCS to Chronophage" is default)
    if args.module == 'crm':
        from pcs_proxy.server_crm import CollaboraServiceProxyLogicCRM
        CollaboraServiceProxyLogic = CollaboraServiceProxyLogicCRM
    else:
        from pcs_proxy.server_chronophage import CollaboraServiceProxyLogicChronophage
        CollaboraServiceProxyLogic = CollaboraServiceProxyLogicChronophage

    settings['do_debug'] = args.debug
    settings['do_record'] = args.record

    # get username and password (supports CLI passage of credos for
    # ease in testing)
    credential_username = args.username
    credential_password = args.password
    apikey = args.apikey

    # make proxy logic provider available globally
    settings['ProxyLogicProvider'] = CollaboraServiceProxyLogic()

    if args.server:
        if args.module == 'crm':
            if not args.api_server:
                print("Please specify an API server for the alternate CRM server")
                exit(1)
            else:
                settings['ProxyLogicProvider'].service_api = args.api_server

        print("Using alternate server: " + args.server)
        if args.api_server:
            print("Using API server: " + args.api_server)
        settings['ProxyLogicProvider'].service_hostname = args.server

    if settings['do_debug']:
        print("Debugging mode enabled!")

    settings['Authenticator'] = Authenticator()

    require_api_key = args.module == "chronophage"

    if (require_api_key and not apikey) or (not require_api_key and not (credential_username and credential_password)):
        # Grab credentials, or get them from the keyring if we can
        settings['Authenticator'].find_in_keyring(settings['ProxyLogicProvider'].service_hostname)

        if settings['Authenticator'].keyring_data:
            debug_log("KEYRING: There are saved credentials in your login keyring, those have been used!")
            if args.module == 'chronophage':
                _, apikey = settings['Authenticator'].keyring_data
            else:
                credential_username, credential_password = settings['Authenticator'].keyring_data
        else:
            if require_api_key:
                apikey = settings['Authenticator'].gather_apikey()
            else:
                credential_username, credential_password = settings['Authenticator'].gather_username_and_password()

    # make credos available to the downline modules
    settings['credo_u'] = credential_username
    settings['credo_p'] = credential_password
    settings['apikey'] = apikey

    settings['lastFail'] = ""
    settings['lastInfo'] = ""

    # setup to start handling connections
    # Listen on 127.0.0.1 specfically to avoid isseus with python not dealing with
    # IPV6 properly
    proxy_handler = partial(CollaboraServiceProxyHandler, CollaboraServiceProxyLogic, settings)

    if args.port:
        proxy_port = args.port
        debug_log(f"Using a custom TCP port ({proxy_port}) for the proxy server")
    else:
        proxy_port = CollaboraServiceProxyLogic.port_TCP()
        debug_log(f"Using the default TCP port ({proxy_port}) for the proxy server")

    pcs_server = http.server.HTTPServer(('127.0.0.1', proxy_port), proxy_handler)

    # perform first-pass authorization if the proxy logic provider supports that
    if settings['ProxyLogicProvider'].get_authentication_type() == AuthenticationType.BASIC:
        settings['ProxyLogicProvider'].reconfirm_basic_credentials()

    # A KeyError exception is thrown if authentication fails, catch this exception
    # and exit with a more friendly message.
    try:
        # perform first-time SID negotiation for proxy logic providers that need it
        settings['ProxyLogicProvider'].session_ensure()
    except Exception as e:
        print(f"Session authentication failed: {e}")
        exit(1)

    # setup baseline credentials if the authentication logic provider supports it
    settings['Authenticator'].set_baseline_credentials(settings['credo_u'], settings['credo_p'])

    # Only saving credentials if the keyring is available and after ensuring the session
    # credentials.
    if settings['Authenticator'].keyring_ok:
        if args.module == 'crm' and not (args.username and args.password):
            user, pwd = settings['credo_u'], settings['credo_p']
        elif args.module == 'chronophage' and not args.apikey:
            user, pwd = os.getlogin(), settings['apikey']
        else:
            user, pwd = None, None

        if user and pwd:
            settings['Authenticator'].save_to_keyring(
                settings['ProxyLogicProvider'].service_hostname,
                user,
                pwd)

    # start handling connections
    print("Ready to serve requests to " + CollaboraServiceProxyLogic.request_URL())
    print("Listening on port " + str(CollaboraServiceProxyLogic.port_TCP()) + "...")

    if return_server:
        return pcs_server
    else:
        pcs_server.serve_forever()


if __name__ == '__main__':
    main()
